<?php

namespace AgeGateLogs\Util;

class Form
{
    /**
     * Sanitize the input data.
     *
     * @param array $input
     * @param array $fields
     * @param bool $utf8_encode
     *
     * @return array
     */
    public function sanitize(array $input, array $fields = [], bool $utf8_encode = true)
    {
        if (empty($fields)) {
            $fields = array_keys($input);
        }

        $return = [];

        foreach ($fields as $field) {
            if (!isset($input[$field])) {
                continue;
            } else {
                $value = $input[$field];
                if (is_array($value)) {
                    $value = $this->sanitize($value, [], $utf8_encode);
                }
                if (is_string($value)) {
                    if (strpos($value, "\r") !== false) {
                        $value = trim($value);
                    }

                    if (function_exists('iconv') && function_exists('mb_detect_encoding') && $utf8_encode) {
                        $current_encoding = mb_detect_encoding($value);

                        if ($current_encoding != 'UTF-8' && $current_encoding != 'UTF-16') {
                            $value = iconv($current_encoding, 'UTF-8', $value);
                        }
                    }

                    $value = filter_var($value, FILTER_SANITIZE_STRING);
                }

                $return[$field] = $value;
            }
        }

        return $return;
    }

    public function validateDob($dob, $allowEmpty = false)
    {
        $errors = [];
        $rules = [
            'd' => 31,
            'm' => 12,
            'y' => date('Y'),
        ];
        
        foreach ($dob as $key => $value) {
            if ($allowEmpty && empty($value)) {
                continue;
            }
            
            if (empty($value) || !is_numeric($value) || $value > $rules[$key]) {
                $errors[$key] = true;
            }
        }

        return $errors;
    }

    public function calcAge($dob)
    {
        $dob = intval($dob['y']). '-' . str_pad(intval($dob['m']), 2, 0, STR_PAD_LEFT) . '-' . str_pad(intval($dob['d']), 2, 0, STR_PAD_LEFT);

        $from = new \DateTime($dob);
        $to   = new \DateTime('today');
        return $from->diff($to)->y;
    }

    

    public function testAge($age, $target = false)
    {
        if (!$target) {
            $data = new \AgeGateLogs\Util\Data;
            $target = $data->getCoreSettings('restrictions')->min_age;
        }

        return $age >= $target;
    }
}
