<?php

namespace AgeGate\UserRegistration\Integration;

use Asylum\Validation\Validator;
use Asylum\Utility\StringTemplate;
use AgeGate\Common\Settings as CoreSettings;
use AgeGate\UserRegistration\Settings\Settings;

class Woocommerce extends AbstractIntegration
{
    public function __construct()
    {
        add_action('wp_enqueue_scripts', [$this, 'assets']);

        /* Registration */
        add_action( 'woocommerce_register_form', [$this, 'registrationFields']);
        add_action( 'woocommerce_register_post', [$this, 'validateRegistration'], 10, 3 );
        add_action( 'woocommerce_created_customer', [$this, 'store']);

        /* My account settings */
        add_action( 'woocommerce_edit_account_form', [$this, 'accountFields'] );
        add_action('woocommerce_save_account_details_errors', [$this, 'validateAccount']);
        add_action('woocommerce_save_account_details', [$this, 'store']);

        /* Checkout */
        /**
         * Add the field to the checkout
         */
        add_action( 'woocommerce_after_checkout_billing_form', [$this, 'checkoutFields']);//]woocommerce_before_order_notes
        add_action('woocommerce_checkout_process', [$this, 'validateCheckout']);
        add_action('woocommerce_after_checkout_registration_form', [$this, 'registrationFields']);
    }

    public function assets()
    {
        if (function_exists('is_checkout') && is_checkout()) {
            wp_enqueue_script( 'age-gate-user-registration-checkout', AGE_GATE_USER_REGISTRATION_URL . 'dist/main.js', [], AGE_GATE_USER_REGISTRATION_VERSION, true );
        }
    }

    public function registrationFields()
    {
        $settings = Settings::getInstance();
        if (is_checkout()) {
            if ($settings->restrictRegisterWoocommerce && !$settings->restrictCheckoutWoocommerce) {
                $this->render($settings, CoreSettings::getInstance());
            }
        } else {
            if ($settings->restrictRegisterWoocommerce) {
                $this->render($settings, CoreSettings::getInstance());
            }
        }
    }

    public function accountFields()
    {
        $settings = Settings::getInstance();

        if ($settings->restrictRegisterWoocommerce) {
            $this->render($settings, CoreSettings::getInstance());
        }
    }

    public function checkoutFields()
    {
        $settings = Settings::getInstance();

        if ($settings->restrictCheckoutWoocommerce) {
            $this->render($settings, CoreSettings::getInstance(), false);
        }
    }


    /**
     * Undocumented function
     *
     * @param [type] $username
     * @param [type] $email
     * @param [type] $errors
     * @return void
     */
    public function validateRegistration($username, $email, $errors)
    {
        $settings = Settings::getInstance();

        if (!$settings->restrictRegister) {
            return;
        }

        $core = CoreSettings::getInstance();

        $validation = new Validator;

        $data = $validation->sanitize($_POST);

        $minYear = apply_filters('age_gate/form/min_year', 1900);
        $maxYear = apply_filters('age_gate/form/max_year', date('Y'));

        $data['age_gate']['dob'] = (new StringTemplate())->render('{y}-{m}-{d}', $data['age_gate']);

        $rules = [
            'age_gate.d' => 'required|numeric|min_len,1|max_len,2|max_numeric,31',
            'age_gate.m' => 'required|numeric|min_len,1|max_len,2|max_numeric,12',
            'age_gate.y' => 'required|numeric|min_len,4|max_len,4|min_numeric,'. $minYear .'|max_numeric,' . $maxYear,
            'age_gate.dob' => 'required|date|min_age,' . ($settings->registerAge ?: $core->defaultAge)
        ];

        $validation->validation_rules($rules);

        $validation->run($data);

        if ($validation->errors()) {
            $error =  (count($validation->get_errors_array()) === 1) ? $settings->registrationError : $settings->invalidError;
            $errors->add('toyoungerror', sprintf($error, $settings->registerAge ?: $core->defaultAge));
        };

        return $errors;
    }

    public function validateAccount($errors)
    {
        $settings = Settings::getInstance();

        if (!$settings->restrictRegister) {
            return;
        }

        $core = CoreSettings::getInstance();

        $validation = new Validator;

        $data = $validation->sanitize($_POST);

        $minYear = apply_filters('age_gate/form/min_year', 1900);
        $maxYear = apply_filters('age_gate/form/max_year', date('Y'));

        $data['age_gate']['dob'] = (new StringTemplate())->render('{y}-{m}-{d}', $data['age_gate']);

        $rules = [
            'age_gate.d' => 'required|numeric|min_len,1|max_len,2|max_numeric,31',
            'age_gate.m' => 'required|numeric|min_len,1|max_len,2|max_numeric,12',
            'age_gate.y' => 'required|numeric|min_len,4|max_len,4|min_numeric,'. $minYear .'|max_numeric,' . $maxYear,
            'age_gate.dob' => 'required|date|min_age,' . ($settings->registerAge ?: $core->defaultAge)
        ];

        $validation->validation_rules($rules);

        $validation->run($data);

        if ($validation->errors()) {
            $error =  (count($validation->get_errors_array()) === 1) ? $settings->registrationError : $settings->invalidError;
            $errors->add('toyoungerror', sprintf($error, $settings->registerAge ?: $core->defaultAge));
        };

        return $errors;
    }

    /**
     * Store registration data
     *
     * @param int $customerId
     * @return void
     */
    public function store( $customerId ) {

        $settings = Settings::getInstance();

        if (!$settings->restrictRegister) {
            return;
        }

        if (!$settings->storeDob) {
            delete_user_meta( $customerId, 'u_db');
            delete_user_meta( $customerId, 'user_dob');
            return;
        }

        $core = CoreSettings::getInstance();

        $validation = new Validator;
        $data = $validation->sanitize($_POST);

        if (!isset($data['age_gate'])) {
            return;
        }

        $validated = Validator::is_valid($data['age_gate'], array(
            'd' => 'required|numeric|min_len,2|max_len,2|max_numeric,31',
            'm' => 'required|numeric|min_len,2|max_len,2|max_numeric,12',
            'y' => 'required|numeric|min_len,4|max_len,4|max_numeric,' . date('Y'),
        ));


        if ($validated === true) {
            update_user_meta( $customerId, 'u_db', $data['age_gate']['y'] . '-' . $data['age_gate']['m'] . '-' . $data['age_gate']['d'] );
            update_user_meta( $customerId, 'user_dob', $data['age_gate'] );
        }
    }

    public function validateCheckout()
    {
        $settings = Settings::getInstance();

        if (!$settings->restrictCheckoutWoocommerce && !$settings->restrictRegisterWoocommerce) {
            return;
        }

        $validation = new Validator;
        $data = $validation->sanitize($_POST);

        if (!$settings->restrictCheckoutWoocommerce && empty($data['createaccount'])) {
            return;
        }

        if (!isset($data['age_gate'])) {
            return;
        }

        $validated = Validator::is_valid($data['age_gate'], array(
            'd' => 'required|numeric|min_len,2|max_len,2|max_numeric,31',
            'm' => 'required|numeric|min_len,2|max_len,2|max_numeric,12',
            'y' => 'required|numeric|min_len,4|max_len,4|max_numeric,' . date('Y'),
        ));


        if ($validated !== true) {
            $core = CoreSettings::getInstance();
            $error = !$settings->restrictCheckoutWoocommerce && $settings->restrictRegisterWoocommerce ? $settings->registrationError : $settings->checkoutError;
			wc_add_notice( sprintf($error, $settings->age ?: $core->defaultAge), 'error');
        }
    }

}
